/* Buttons To Press btp-6.4.1;  Enrique Vidal, March-2003.
   - New code for LED flashing (only works with 8.1+ kernels),
   - A new keyword "copyreset" to specify a "copy-without-sending" default,
   - All the string malloc code now in a generic function: getStrMem.

   Buttons To Press btp-6.4.0;  Enrique Vidal, Martin Samuelson, Sep-2002).
   - Move all LIRC-dependent code to rc.cxx,
   - Improved multiple layouts or "views" using Fl_Group,
   - Correct auto-scaling with the new multiple layouts code,
   - Improved beeping code,
   - Fixed a few memory leaks.

   Buttons To Press btp-6.3.3;  Enrique Vidal, evidal@iti.upv.es, Apr, 2002.
   - button labels are now folded if their length doesn't fit,
   - "copy-without-sending" function,
   - visual and acoustic indicators of "sending activity",
   - unlimited length commands and macros,
   - multiple layouts.

   Buttons To Press btp-6.3.2;  Enrique Vidal, evidal@iti.upv.es, Feb, 2002.
   - Support for using the AgVr3r hardbuttons.

   Buttons To Press btp-6.3.1;  Martin Samuelsson, sam@home.se, 2002-01-02
   - A mostly-no-frills FLTK GUI for rc, made specially for use with the
     Agenda PDA by Agenda Computing, with support for macros and timers.

   Buttons To Press btp-6.0;  Ronald Rael Harvest, 
                              six@thevillage.dhs.org, 2001-11-19
   - A modification of Buttons To Press 1.0 to make it read irremote
     config files.

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   /Sam
*/
#include <X11/Xlib.h>
#include <X11/keysym.h>

#include <FL/Fl.H>
#include <FL/Fl_Box.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Window.H>
#include <FL/Fl_Output.H>

// #include <FL/Fl_Double_Window.H>

#include <Flek/Fl_App_Window.H>
#include <flpda/Widget_Factory.h>

#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/un.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <errno.h>
#include <time.h>

#include "btp.h"
#include "rc.h"
#include "btpErrors.h"

/*------------------------------------------------------------- Dimensions */
// Since they are nedded by btpErrors.h, these are now in btp.h
// #define D_WIDTH  160
// #define D_HEIGHT 240
// #define G_WIDTH  160
// #define G_HEIGHT 220
// #define R_WIDTH  160
// #define R_HEIGHT 100
// #define R_BORDER 2
// #define R_BOTTOM 16

/*-------------------------------------------- Hardware button codes, etc. */
#define UnknownInd  -1
#define Shift_LInd   0
#define Shift_RInd   1
#define LeftInd      2
#define RightInd     3
#define Page_UpInd   4
#define Page_DownInd 5

#define Shift_L   "Shift_L"
#define Shift_R   "Shift_R"
#define Left      "Left"
#define Right     "Right"
#define Page_Up   "Page_Up"
#define Page_Down "Page_Down"

#ifdef AGVR3
#define Shift_R_Code  24       // Shift_R code for Agenda
#else
#define Shift_R_Code  62       // Shift_R code for host
#endif
Display *Xdisp;                // For changing the Shift_R code


/*----------------------------- forward declarations (function prototypes) */
int  loadLayoutFile(char *fname);
void cb_resetCopyBlink();
char *getStrMem(char *buff, int bytes);

/*------------------------------------------------- For the sending rutine */
// extern int rcsend(char *command, int fd);              // now in "rc.h"
// int fd; // The socket.                                // Now in rc.cxx

/*------------------------------------------------- Global vars and tables */
FILE *consoleFile;             // For the LEDs
Display *XdispBeep;            // For the beeps

#define maxIRs 16              // Maximum number of remote layouts (.ini files)
char *fname[maxIRs];           // Filenames of the .ini files
char *layoutID[maxIRs];        // Layout ID's
int numIRs;                    // Actual  number of remote layouts (.ini files)
int activeIR=0;                // Presently active  remote layout

// Fl_Double_Window *btpwindow;// doublebuffer Window for the layouts;
                               // pity: does not seem supported by Widg.Factory
Fl_App_Window *btpwindow;      // Winodw for the layouts
Fl_Button *curr;               // Toolbar label  for the current layout 
Fl_Button *next;               // Toolbar button for the next    layout

Fl_Button *beeps;              // Toolbar button to enable/disable beeps
int wantBeeps=1;               // beeping level (0..2)

void *prevData=NULL;           // Global pointer to last-button-press data.
int wantCopy=0;                // Flag for the CopyBlink button.
Fl_Button *CopyBlink;          // Used for hardbuttons copy-without-sending,
                               // and as an indicator that rcsend is working.

int show_toolbar = 1;          // show_toolbar flag

#define maxHDbutt 6            // Number of available hardbuttons
char *hdcomm[maxHDbutt];       // The hadrbutton commands

char *copyReset=NULL;          // The copyreset command
char *copyResetLabel=NULL;     // The copyreset label

/*-------------------------------------------------------------------------*/
/*-------------------------------------- Toolbar curr/next button handling */
void labelNextCurr() {
  int j;  
  char *id1=NULL, *id2=NULL;  id1 = getStrMem(id1,5); id2 = getStrMem(id2,5);

  if (! show_toolbar) return;
  id1[0]='\0'; strncat(id1, layoutID[activeIR],4);
  curr->label(id1); curr->show(); curr->redraw();

  j=activeIR+1; if (j>numIRs-1) j=0;

  id2[0]='\0'; strncat(id2,">",1); 
  strncat(id2, layoutID[j],3); next->label(id2); next->show(); next->redraw();
}

void cb_curr(Fl_Widget *widget, void *data) {
  Fl_Button *b = CopyBlink;
  b->label("Copy"); b->clear(); b->show(); b->redraw();
  prevData=NULL; wantCopy=0;
}

void cb_next(Fl_Widget *widget, void *data) {
  Fl_Button *b = CopyBlink;
  activeIR++; if (activeIR>numIRs-1) activeIR=0;
  b->label("Copy"); b->clear(); b->show(); b->redraw();
  prevData=NULL; wantCopy=0;

  loadLayoutFile(fname[activeIR]);  labelNextCurr();  cb_resetCopyBlink();
}

void cb_layout(Fl_Widget *w, void *data) {
  int i; Fl_Button *b;

  i=0; while (i<numIRs && strcmp(layoutID[i],(char *)data)) i++; 
  if (i < numIRs) {
    activeIR=i;
    b=CopyBlink;  b->label("Copy"); b->clear(); b->show(); b->redraw();
    prevData=NULL; wantCopy=0;

    loadLayoutFile(fname[activeIR]);  labelNextCurr();  cb_resetCopyBlink();
  }
}

/*------------------------------------------------------------------- Beeps */
                    /* From beep.c, (C) 2001 James Fenn <jfenn@uklinux.net> */
void beep(int pitch, int duration, int permLevel) {
//Display *XdispBeep;                            // Global; assumedly open.
  XKeyboardControl vals;

  if (wantBeeps < permLevel) return;

  vals.bell_pitch = pitch; vals.bell_duration = duration;
  vals.bell_percent = 100;

  XChangeKeyboardControl(
    XdispBeep, (KBBellPitch | KBBellDuration | KBBellPercent), &vals
  );
  XBell(XdispBeep, 100);  XFlush(XdispBeep);
}
void cb_beeps(Fl_Widget *widget, void *data) {
  wantBeeps = (wantBeeps+1)%3;
  if      (wantBeeps==0) {beeps->label("NoBps");  beeps->set();}
  else if (wantBeeps==1) {beeps->label("Beeps");  beeps->clear();}
  else                   {beeps->label("AllBps"); beeps->set();}
}

/*------------ Standard button callback, copy-without-sending and blinking */
void cb_resetCopyBlink() {    // Timeout to reset CopyBlink toolbar button 
  // Remove timeout, just in case we are called from curr/next button cb's
  Fl::remove_timeout((Fl_Timeout_Handler)cb_resetCopyBlink); 
  Fl_Button *b=CopyBlink;
  b->label(copyResetLabel); b->clear(); b->show(); b->redraw(); 
  prevData=copyReset;  wantCopy=0;
//fprintf(stderr,"--> Timeout exhausted!\n"); 
}
void CopyBlinkTimeout() {
#define CopyBlinkResetTime 30.0
  Fl::remove_timeout(                 (Fl_Timeout_Handler)cb_resetCopyBlink);
  Fl::add_timeout(CopyBlinkResetTime, (Fl_Timeout_Handler)cb_resetCopyBlink);
}

void cb_copy(Fl_Widget *widget, void *data) {
  if (show_toolbar) {
     Fl_Button *b=CopyBlink;
     b->label("Copy"); b->set(); b->show(); b->redraw();
  }
  prevData=NULL; wantCopy=1; CopyBlinkTimeout();
}

void ledFlash(int autoStop, int offPeriod, int onPeriod, int blinkNum) {
#ifdef AGVR3
  if (consoleFile == NULL) return;
  fprintf(consoleFile, "\033[16;%d]\033[17;%d]\033[19;%d]\033[18;%d]\n", 
                               onPeriod, offPeriod, blinkNum, autoStop);
#endif
}
void ledFlashOff() {
#ifdef AGVR3
  if (consoleFile == NULL) return; fprintf(consoleFile, "\033[18;0]\n");
#endif
}

void blinker(int OnOff, Fl_Button *b) {
  static char *label;
//fprintf(stderr,"--> %s\n", b->label()); 
  if (OnOff) {
     ledFlash(1,3,1,0);  label = strdup(b->label());
     beep(1200, 60, 1);  b->label("))))"); b->set();
  }
  else {b->label(label);  b->clear();  beep(1200, 200, 1);  ledFlashOff();}
  b->redraw(); Fl::flush();
}

void cb_button(Fl_Widget *w, void *data) {
  Fl_Button *b = (Fl_Button *)w, *c=CopyBlink;
//fprintf(stderr,"--> %s\n    %s, %d\n",(char*)data,(char*)prevData,wantCopy);

  prevData = data;             // just copy data to prevData;  don't send the 
  if (wantCopy) wantCopy=0;    // command if wantCopy, or if event is "drag".
  else if (Fl::event_is_click() || (w == c)) {
    #define maxLength 50          // This should rather be done by estimating 
    int isLong = (strlen((char *)data) > maxLength);        // macro timing!!
    if (isLong) blinker(1, b);
    rcsend((char *)data);
    ledFlash(3,0,1,1);
//  beep(1000, 8, 2);   // all buttons would beep
    if (isLong) {blinker(0, b); usleep(600000);} 
    else if (show_toolbar) c->clear();
  }
//fprintf(stderr,"--->   %s\n", b->label()); 
  if (show_toolbar) {
     c->label(b->label());  // Change Copy-label to remind what's been copied.
     if (w != c) c->set(); c->redraw(); Fl::flush();
  }
  CopyBlinkTimeout();
}

/*----------------------------------------- sending from hardware buttons */
// void doubleClick(); // now in album.h
void rcHdSend(char *data) {
//fprintf(stderr,"===> %s\n", (char *)data);
  if (!strncasecmp(data, ".prev", 5)) {
     if (prevData != NULL)
        cb_button(CopyBlink, prevData); beep(300,60,1);
  }
  else { rcsend(data); ledFlash(3,0,1,1); }
}

/*------------------------------------------------------- X Error Handling */
int Xerr(Display *displ, XErrorEvent *z) {      //  Catch possible errors  of
char buf[100];                                 // changing Xfree key mappings
  XGetErrorText(displ, z->error_code, buf, 100);
  btpStderr("Xfree ERROR: %s\n", buf);
  return z->error_code;
}

/*------------------------------------------------------------------- Done */
void cb_done(Fl_Widget *widget, void *data) {
KeySym ks;

  Fl_Window* window = reinterpret_cast<Fl_Window*>(data);
  window->hide();
  XCloseDisplay(XdispBeep);
  fclose(consoleFile);

#ifdef AGVR3
  ks = XK_Shift_L;
#else
  ks = XK_Shift_R;
#endif
  Xdisp=XOpenDisplay(NULL); 
  XChangeKeyboardMapping(Xdisp, Shift_R_Code, 1, &ks, 1);
  XCloseDisplay(Xdisp);                      // Restores the standard Shift_R

  endRc();
  exit(EXIT_SUCCESS); 
}

/*------------------------------------------ To load all layouts initially */
#define iniLoadTime 0.5
void cb_timeout() {
  static int i=0;
  Fl::remove_timeout((Fl_Timeout_Handler)cb_timeout);
  cb_next(NULL, NULL); i++;
  if (i<numIRs) Fl::add_timeout(iniLoadTime, (Fl_Timeout_Handler)cb_timeout);
}

/*------------------------- Get memory for a string in a controlled manner */
char *getStrMem(char *buff, int bytes) {
  if (buff != NULL) free(buff);
  if ((buff = (char *)malloc(bytes)) == NULL) 
     {btpStderr("Out of memory: %d\n", bytes); exit(EXIT_FAILURE);}
  return(buff);
}

/*---------------------------------------------------- Define a new Layout */
int loadLayoutFile(char *fname) {
FILE *rcfile;
char layout_id[64];
char t[1024];                                   // The .init file read buffer
int labelsize = 10;
int layoutgood = 0;
int bytes, i=0;
int curw, curh;
Fl_Widget *b;
static int first=1;
static Fl_Group *btpgroup;

  if (copyReset != NULL) free(copyReset);  copyReset=NULL;
  if (copyResetLabel != NULL) free(copyResetLabel);
  copyResetLabel = strdup("Copy");  prevData=NULL;

  //fprintf(stderr,"llf: %s\n", fname);

  //if (btpgroup != NULL) btpgroup->~Fl_Group();
  if (first) {
    //fprintf(stderr,"llf: first=true\n");
    btpgroup = new Fl_Group(
      btpwindow->x(), btpwindow->y(), btpwindow->w(), btpwindow->h()
    );
    btpgroup->window()->resizable(btpgroup);
  } else {
    //fprintf(stderr,"llf: first=false\n");
    int k = btpgroup->children();
    //for (i=0; i<k; i++) {
    for (i=k-1; i>=0; i--) {
      b = btpgroup->child(i);
      // fprintf(stderr, "removing i=%d x=%4d y=%4d label=%s\n", 
      //                           i, b->x(), b->y(), b->label());

      b->hide(); btpgroup->remove(b); // Bug?: remove does not free anything!
                                                // let's do it one by one ...
      free(b->user_data()); free((char *)b->label());
      // b->~Fl_Widget();  // This destructor seems to have no effect either!
      // delete(b->window());                   // ?? Seems to have no efect.
      delete(b);

      curw=btpgroup->w(); curh=btpgroup->h();
    }
    // btpgroup->init_sizes();
  }

  sprintf(layout_id, "%d", activeIR);                    // default layout ID
  if ((rcfile = fopen(fname, "r"))) {
    while (fgets(t, 1024, rcfile)) {

      if (!strncasecmp(t, "layout", 6)) {          // Parse and set layout ID
        char *parseable = index(t, '=') + 1;
        strcpy(layout_id, strtok(parseable, ",\n"));
      }
      if (!strncasecmp(t, "toolbar", 7)) {
        char *parseable = index(t, '=') + 1;
        show_toolbar = !strncasecmp(parseable, "show", 4);
      }
      if (!strncasecmp(t, "labelsize", 9)) {
        char *parseable = index(t, '=') + 1;
        labelsize = atoi(strtok(parseable, ",\n"));
      }
      // --------------------------------------------------------------------
      if (!strncasecmp(t, "copyreset", 9)) { // Parse copyreset specification
        char *parseable = index(t, '=') + 1;
        char *label = strdup(strtok(parseable, ","));
        char *remote = strdup(strtok(NULL, ","));
        char *remoteCode = strdup(strtok(NULL, "\n"));

        bytes = strlen(remote)+strlen(remoteCode)+3;
        copyReset = getStrMem(copyReset, bytes);
        sprintf(copyReset, "%s %s", remote, remoteCode);

        if (copyResetLabel != NULL) free(copyResetLabel);
        copyResetLabel = strdup(label);
        free(remote); free(remoteCode); free(label);
        //fprintf(stderr, "--- - %s - %s -\n", copyResetLabel, copyReset);
      }
      // --------------------------------------------------------------------
      if (!strncasecmp(t, "hardbutton", 10)) {           // Parse hardbuttons
        int j=UnknownInd;
        char *parseable = index(t, '=') + 1;
        char *key = strdup(strtok(parseable, ","));
        char *remote = strdup(strtok(NULL, ",\n"));
        char *remoteCode;
        if (!strncasecmp(remote, ".prev", 5)) remoteCode = strdup("");
        else remoteCode = strdup(strtok(NULL, "\n"));

        if      (strcmp(key, Shift_L)   == 0) j=Shift_LInd;
        else if (strcmp(key, Shift_R)   == 0) j=Shift_RInd;
        else if (strcmp(key, Left)      == 0) j=LeftInd;
        else if (strcmp(key, Right)     == 0) j=RightInd;
        else if (strcmp(key, Page_Up)   == 0) j=Page_UpInd;
        else if (strcmp(key, Page_Down) == 0) j=Page_DownInd; 
  
        if (j>=0 && j<maxHDbutt) {
           bytes = strlen(remote)+strlen(remoteCode)+3;
           hdcomm[j] = getStrMem(hdcomm[j], bytes);
           sprintf(hdcomm[j], "%s %s", remote, remoteCode);
        }
        else btpStderr("Hard button unknown: %s\n", key);
        //fprintf(stderr, "--- %d %s - %s-\n", j, key, hdcomm[j]);
        free(remote); free(remoteCode); free(key);
      } // -------------------------------------------- End parse hardbuttons
  
      // ----------------------------- Parse normal and layout buttons
      if (!strncasecmp(t, "button", 6)) {
        char *parseable = index(t, '=') + 1;
        int x = atoi(strtok(parseable, ","));
        int y = atoi(strtok(NULL, ","));
        int width = atoi(strtok(NULL, ","));
        int height = atoi(strtok(NULL, ","));
        char *label = strdup(strtok(NULL, ","));
        char *remote = strdup(strtok(NULL, ","));
        char *remoteCode = strdup(strtok(NULL, "\n"));

        bytes = strlen(remote)+strlen(remoteCode)+3;
        char *command; command = getStrMem(NULL, bytes);
        sprintf(command, "%s %s", remote, remoteCode);
        
	if(!first) {
	  x = (x*curw)/G_WIDTH;	  width  =  (width*curw)/G_WIDTH;  
	  y = (y*curh)/G_HEIGHT;  height = (height*curh)/G_HEIGHT;
	}

	Fl_Button *b = new Fl_Button(x, y, width, height, label);
	b->labeltype(FL_SYMBOL_LABEL);
        b->align(FL_ALIGN_WRAP | FL_ALIGN_CLIP);
        b->labelsize(labelsize);
        if (!strncasecmp(remote, "layout", 6))          // Call layout button
	     {b->callback(cb_layout, remoteCode); free(command); command=NULL;}
        else {b->callback(cb_button, command); free(remoteCode);}
        free(remote);
        if (!first) btpgroup->add(b); b->redraw(); // b->show(); 
      }
    } // END: while (fgets(t, 1024, rcfile))
    fclose(rcfile); 
  } // END: if ((rcfile = fopen(fname, "r")))

  if (first) btpgroup->end();

//btpgroup->show();  btpgroup->redraw();  btpwindow->redraw(); 

  first=0;
  layoutgood = (strcmp(layout_id, layoutID[activeIR]) == 0);
  return layoutgood;
}


/*------------------------------------------------------------------------ */
/*------------------------------------------------------------------- Main */
int main(int argc, char** argv) {
int layoutgood = 0;
int i=0, j;

    if (!Fl::args(argc,argv,i)) Fl::fatal(Fl::help);
    fname[0] = (i < argc) ? strdup(argv[i]) : strdup("remote.ini");

    numIRs= (argc>1) ? argc-1 : 1;
    if (numIRs > maxIRs) {
      btpStderr("Too many IR .ini files (%d > maxIRs)\n", numIRs);
      numIRs=maxIRs;
    }
    for (j=0; j<maxHDbutt; j++) hdcomm[j] = strdup("");
    
    for (i=j=0; ((i<numIRs) && (i<maxIRs)); i++)    {     FILE *rcfile; 
      if (argc>1) fname[j] = argv[i+1];
      //fprintf(stderr,"%d: %s  %s\n", i+1, argv[i+1], fname[i]);
      
      if ((rcfile = fopen(fname[j], "r"))) { char t[64];    // Set layout IDs
         layoutID[j] = getStrMem(layoutID[j], 2);
         sprintf(layoutID[j], "%d", j);                         // Default ID

         fgets(t, 64, rcfile);
         if (!strncasecmp(t, "layout", 6)) {              // Parse and set ID
            char *parseable = index(t, '=') + 1;
            layoutID[j] = strdup(strtok(parseable, ",\n"));
         }
         j++;
      }
      else btpStderr("Could not open %s\n", fname[j]);       // Mild error...
      //fprintf(stderr,"%d,%d: %s  %s\n", i,j-1, fname[j-1], layoutID[j-1]);
    }
    i=0; numIRs=j;

    // btpwindow = new Fl_Double_Window(0,0,160,240,"BTP 6.3.3");
    // Double_Window is much better, but not seems supported by WidgetFactory
    btpwindow = WidgetFactory::new_window("BTP 6.4.1");
    layoutgood = loadLayoutFile(fname[0]);
    if (show_toolbar) {
       Fl_Dockable_Window* toolbar = WidgetFactory::new_toolbar();
       Widget_Factory::new_button("Done",              cb_done, btpwindow);
       beeps = Widget_Factory::new_button("Beeps",    cb_beeps, btpwindow);
       CopyBlink = Widget_Factory::new_button("Copy",  cb_copy, btpwindow);
       CopyBlink->labeltype(FL_SYMBOL_LABEL);
       curr=Widget_Factory::new_button("",             cb_curr, btpwindow);
       next=Widget_Factory::new_button("",             cb_next, btpwindow);
       btpwindow->add_dockable(toolbar, 1);
       toolbar->end();
    }
    btpwindow->end(); btpwindow->show();

    labelNextCurr();                        // label curr/next toolbar button

    if (layoutgood) startRc();
    else  btperror("Layout error!\nCould not open remote.ini\n", 1);

    
    XSetErrorHandler(Xerr);
    KeySym ks = XK_Shift_R;
    Xdisp=XOpenDisplay(NULL);                       // Set the Shift_R button
    XChangeKeyboardMapping(Xdisp, Shift_R_Code, 1, &ks, 1);
    XCloseDisplay(Xdisp);
    XdispBeep=XOpenDisplay(NULL);   // For the beeps; to be closed by cb_done
    consoleFile = fopen("/dev/console", "w");             // for LED flashing
    if (consoleFile == NULL) {                                           
       btpStderr("Error opening /dev/console: %s.\n", strerror(errno));
    }

    Fl::add_timeout(iniLoadTime, (Fl_Timeout_Handler)cb_timeout);

    while (Fl::wait()) {
      int k = Fl::event_key();   // Bad idea, needed for AgVr3 FL_Left/Right!
      if (k==0) continue;
//    fprintf(stderr, ">>>> %u\n", k);
      if      (Fl::event_key(FL_Shift_L) && strlen(hdcomm[Shift_LInd]))
              {rcHdSend(hdcomm[Shift_LInd]);}
      else if (Fl::event_key(FL_Shift_R) && strlen(hdcomm[Shift_RInd]))
              {rcHdSend(hdcomm[Shift_RInd]);}
//    else if (Fl::event_key(FL_Left))      // Pity: does'nt work on AgVr3 !?
      else if (k==FL_Left && strlen(hdcomm[LeftInd]))
              {rcHdSend(hdcomm[LeftInd]);}
//    else if (Fl::event_key(FL_Right))     // Pity: does'nt work on AgVr3 !?
      else if (k==FL_Right && strlen(hdcomm[RightInd]))
              {rcHdSend(hdcomm[RightInd]);}
      else if (Fl::event_key(FL_Page_Up) && strlen(hdcomm[Page_UpInd]))
              {rcHdSend(hdcomm[Page_UpInd]);}
      else if (Fl::event_key(FL_Page_Down) && strlen(hdcomm[Page_DownInd]))
              {rcHdSend(hdcomm[Page_DownInd]);} 
    }
}
